## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set( # nolint: extraction_operator_linter.
    collapse = TRUE,
    comment = "#>"
)

## ----setup--------------------------------------------------------------------
library(tidyverse)
library(drugfindR)

## ----install------------------------------------------------------------------
# BiocManager::install("drugfindR") # nolint: nonportable_path_linter.

## ----load_signature-----------------------------------------------------------
# Load the signature from the paper

diffexp <- read_tsv(
    system.file("extdata", "dCovid_diffexp.tsv",
        package = "drugfindR"
    )
)

# Take a look at the signature

head(diffexp) |>
    knitr::kable()

## ----prepareSignature---------------------------------------------------------
# Prepare the signature for analysis
# The only thing that is different from the defaults is the gene_column
# so we will specify that and rely on defaults for others.

signature <- prepareSignature(diffexp,
    geneColumn = "hgnc_symbol"
)

# Take a look at the signature

head(signature) |>
    knitr::kable()

## ----filterSignatureUp--------------------------------------------------------
# Filter the signature to only include genes that are upregulated by at least
# 1.5 logFC

filteredSignatureUp <- filterSignature(signature,
    direction = "up",
    threshold = 1.5
)

filteredSignatureUp |>
    head() |>
    knitr::kable()

## ----filterSignature_dn-------------------------------------------------------
# Filter the signature to only include genes that are downregulated by at least
# 1.5 logFC
filteredSignatureDn <- filterSignature(signature,
    direction = "down",
    threshold = 1.5
)

filteredSignatureDn |>
    head() |>
    knitr::kable()

## ----getConcordants-----------------------------------------------------------
# Get the concordant signatures for the upregulated signature

upConcordants <- getConcordants(filteredSignatureUp, ilincsLibrary = "CP")

upConcordants |>
    head() |>
    knitr::kable()

# Get the concordant signatures for the downregulated signature

dnConcordants <- getConcordants(filteredSignatureDn, ilincsLibrary = "CP")

dnConcordants |>
    head() |>
    knitr::kable()

## ----consensusConcordants-----------------------------------------------------
# Get the consensus concordant signatures for the upregulated signature

consensus <- consensusConcordants(upConcordants, dnConcordants,
    paired = TRUE, cutoff = 0.2
)

consensus |>
    head() |>
    knitr::kable()

## ----investigateSignature-----------------------------------------------------
investigated <- investigateSignature(diffexp,
    outputLib = "CP", filterThreshold = 1.5,
    geneColumn = "hgnc_symbol", logfcColumn = "logFC",
    pvalColumn = "PValue"
)

investigated |>
    head() |>
    knitr::kable()

## ----sessionInfo--------------------------------------------------------------
devtools::session_info()

