## ----setup, include=FALSE-----------------------------------------------------
# Set chunk options: suppress echo, messages, and warnings in code output
knitr::opts_chunk$set(echo = TRUE, message = FALSE, warning = FALSE)

## ----load-cosmx---------------------------------------------------------------
library(SpaceTrooper)
library(ggplot2)

cosmxFolder <- system.file(file.path("extdata", "CosMx_DBKero_Tiny"), 
                        package="SpaceTrooper")
spe <- readCosmxSPE(cosmxFolder, sampleName="DBKero_CosMx")

spe

## ----load-xenmer--------------------------------------------------------------
xeniumFolder <- system.file( "extdata", "Xenium_small", package="SpaceTrooper")
xen <- readXeniumSPE(xeniumFolder, computeMissingMetrics=TRUE,
                        keepPolygons=TRUE)

merfishFolder <- system.file("extdata", "Merfish_Tiny", package="SpaceTrooper")
mer <- readMerfishSPE(merfishFolder, boundariesType="parquet",
                        computeMissingMetrics=TRUE, keepPolygons=TRUE)

## ----plot-fovs----------------------------------------------------------------
# to check misalignment
plotCellsFovs(spe, size = 3, alpha = 0.7)

## ----fov-correction-----------------------------------------------------------
# code line for shift correction
# metadata(spe)$fov_positions$y_global_px <- metadata(spe)$fov_positions$y_global_px - 4256

## ----load-poly,  message = TRUE-----------------------------------------------
# polygon loading
spe <- readAndAddPolygonsToSPE(spe, boundariesType="csv")

## ----cosmx-analysis-qc,  message = TRUE---------------------------------------
spe <- spatialPerCellQC(spe, rmZeros=TRUE,
        negProbList=c("NegPrb", "Negative", "SystemControl"))

colnames(colData(spe))

## ----cosmx-analysis-score,  message = TRUE------------------------------------
set.seed(1713)

spe <- computeQCScore(spe)

summary(spe$QC_score)

## ----cosmx-analysis-score2, message = TRUE------------------------------------
spe <- computeQCScoreFlags(spe, qsThreshold=0.5)

table(spe$low_qcscore)

## ----plot-hist----------------------------------------------------------------
# view quantitative metric distribution
plotMetricHist(spe, metric = "QC_score")

## ----plot-centroids-labels----------------------------------------------------
labf <- system.file(file.path("extdata", "CosMx_DBKero_Tiny",
                            "labels_tiny.tsv"), package="SpaceTrooper")
labs <- read.table(file=labf, sep="\t", header=TRUE)

spe$labels <- as.factor(labs[match(spe$cell_id, labs$cell_id),]$label)
spe$labels_colors <- as.factor(labs[match(spe$cell_id, labs$cell_id),]$lab_color)

plotCentroids(spe, colourBy="labels", size=3, palette="labels_colors")

## ----plot-polygons-fov-1------------------------------------------------------
plotPolygons(spe, colourBy="log2SignalDensity")
plotPolygons(spe, colourBy="Area_um")
plotPolygons(spe, colourBy="log2Ctrl_total_ratio")
plotPolygons(spe, colourBy="log2AspectRatio")

## ----plot-polygons-fov-2------------------------------------------------------
plotPolygons(spe, colourBy="QC_score") + 
    scale_fill_viridis_c(option = "plasma")

## ----plot-polygons-fov-3------------------------------------------------------
plotPolygons(spe, colourBy="low_qcscore") + 
    scale_fill_manual(values=c("TRUE"="red", "FALSE" = "#c0c8cf"))

## ----cosmx-analysis-score3, message = TRUE------------------------------------
spe <- computeQCScoreFlags(spe, qsThreshold=0.75)

table(spe$low_qcscore)

## ----plot-polygons-fov-4------------------------------------------------------
plotPolygons(spe, colourBy="low_qcscore") + 
    scale_fill_manual(values=c("TRUE"="red", "FALSE" = "#c0c8cf"))

## ----sessionInfo--------------------------------------------------------------
sessionInfo()

