## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup--------------------------------------------------------------------
suppressPackageStartupMessages({
  library(DenoIST)
  library(SpatialExperiment)
  library(ggplot2)
  library(patchwork)
})

## ----load_spe-----------------------------------------------------------------
spe <- readRDS(system.file("extdata", "example_spe.rds", package = "DenoIST"))
spe

## ----load_data----------------------------------------------------------------
tx <- readRDS(system.file("extdata", "tx_sub.rds", package = "DenoIST"))
tx

## ----denoist------------------------------------------------------------------
# DenoIST
result <- denoist(mat = spe, tx = tx, coords = NULL, tx_x = "x_location", tx_y = "y_location", feature_label = "feature_name", distance = 50, nbins = 200, cl = 1)

## ----read_results-------------------------------------------------------------
result <- readRDS(system.file("extdata", "result.rds", package = "DenoIST"))

## ----check_results------------------------------------------------------------
result$adjusted_counts[1:5, 1:5]

## -----------------------------------------------------------------------------
result$memberships[1:5, 1:5]

## -----------------------------------------------------------------------------
result$params[[42]]

## -----------------------------------------------------------------------------
# a custom helper function for plotting
plot_feature_scatter <- function(coords, mat, feature, size = 0.3, output_filename = NULL) {
  # Create a data frame from the coordinates and features
  plot_data <- data.frame(x = coords[, 1], y = coords[, 2], feature = mat[feature,])
  
  # Create the scatterplot
  p <- ggplot(plot_data, aes(x = x, y = y, color = feature)) +
    geom_point(size = size, alpha = 0.5) +  # Make the dots smaller
    theme_minimal() +
    labs(title = feature, x = "X Coordinate", y = "Y Coordinate", color = "Feature") +
    theme(legend.position = "right") +
    scale_color_viridis_c()  # Use a color palette with higher contrast
  
  # Save the plot if an output filename is provided
  if (!is.null(output_filename)) {
    ggsave(output_filename, p, width = 10, height = 8, units = "in", dpi = 300)
  }
  
  # Return the plot
  return(p)
}

## ----fig.height=8, fig.width=10-----------------------------------------------
orig <- plot_feature_scatter(coords = spatialCoords(spe), log(assay(spe)+1), feature = "EPAS1") + ggtitle('Original')
adj <- plot_feature_scatter(coords = spatialCoords(spe), log(result$adjusted_counts+1), feature = "EPAS1") + ggtitle('DenoIST adjusted')
(orig + adj) + plot_layout(guides = "collect")

## ----sessionInfo--------------------------------------------------------------
sessionInfo()

