## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----eval = FALSE-------------------------------------------------------------
# if (!require("BiocManager", quietly = TRUE))
#     install.packages("BiocManager")
# 
# BiocManager::install("MOSClip")

## ----message=FALSE------------------------------------------------------------
library(MOSClip)
library(MultiAssayExperiment)
library(kableExtra)

data("reactSmall")
data("multiOmics")

## -----------------------------------------------------------------------------
availableOmicMethods()

## ----warning = FALSE----------------------------------------------------------
genesToConsider <- row.names(experiments(multiOmics)$exp)

moduleSurv <- lapply(reactSmall, function(g) {
  set.seed(1234)
  fcl = multiOmicsSurvivalModuleTest(multiOmics, g, 
                                     useTheseGenes = genesToConsider)
  fcl
 })

## -----------------------------------------------------------------------------
moduleSummary <- multiPathwayModuleReport(moduleSurv)

## ----echo=FALSE---------------------------------------------------------------
kable(moduleSummary[1:10,-1]) %>%
  kable_styling() %>%
  scroll_box(width = "80%", height = "400px")

## ----fig.height=6, fig.weight=6-----------------------------------------------
plotModuleReport(moduleSurv[["Activation of Matrix Metalloproteinases"]], 
                 MOcolors = c(exp = "red", met = "green", 
                              cnv = "yellow", mut = "blue"))

## ----message=FALSE------------------------------------------------------------
plotModuleInGraph(moduleSurv[["Activation of Matrix Metalloproteinases"]], 
                  pathList = reactSmall, 
                  moduleNumber = 4)

## ----fig.height=6, fig.width=7------------------------------------------------
plotModuleKM(moduleSurv[["Activation of Matrix Metalloproteinases"]], 
             moduleNumber = 4, 
             formula = "Surv(days, status) ~ expPC2 + met2k", 
             paletteNames = "Paired", risk.table = TRUE, inYears = TRUE)

## ----message=FALSE, results='hide', fig.keep='all', fig.height=7, fig.width=7----
additionalA <- colData(multiOmics)
additionalA$status[additionalA$status == 1] <- "event"
additionalA$status[additionalA$status == 0] <- "no_event"
additionalA$PFS <- as.factor(additionalA$status)
additionalA$status <- NULL
additionalA$years <- round(additionalA$days/365.24, 0)
additionalA$days <- NULL

plotModuleHeat(moduleSurv[["Activation of Matrix Metalloproteinases"]], 
               moduleNumber = 4, 
               paletteNames = c(exp = "red", met = "green", 
                                cnv = "yellow", mut = "blue"),
               additionalAnnotations = additionalA, 
               additionalPaletteNames = list(PFS = "violet", years = "teal"), 
               sortBy = c("expPC2", "met2k", "PFS", "years"), 
               withSampleNames = FALSE)

## ----fig.height=9, fig.width=10-----------------------------------------------
runSupertest(moduleSummary, pvalueThr = 0.05, zscoreThr = 0.05, 
             excludeColumns = c("pathway", "module"))

## ----warning=FALSE------------------------------------------------------------
data("multiOmicsTopo")

pathwaySurv <- lapply(reactSmall, function(g) {
  set.seed(1234)
  fcl = multiOmicsSurvivalPathwayTest(multiOmicsTopo, g, 
                                      useTheseGenes = genesToConsider)
  })

## ----fig.width=6, fig.height=4------------------------------------------------
plotMultiPathwayReport(pathwaySurv, 
                       MOcolors = c(exp = "red", mut = "blue", 
                                    cnv = "yellow", met = "green"))

## ----message=FALSE, results='hide', fig.keep='all', fig.height=7, fig.width=7----
plotPathwayHeat(pathwaySurv[["Activation of Matrix Metalloproteinases"]], 
                sortBy = c("expPC1", "cnvPOS", "PFS"), 
                paletteNames = c(exp = "red", met = "green",
                                 mut = "blue", cnv = "yellow"), 
                additionalAnnotations = additionalA, 
                additionalPaletteNames = list(PFS = "violet", years = "teal"), 
                withSampleNames = FALSE)

## ----fig.height=9, fig.width=10-----------------------------------------------
plotPathwayKM(pathwaySurv[["Activation of Matrix Metalloproteinases"]], 
              formula = "Surv(days, status) ~ expPC1 + cnvPOS", 
              paletteNames = "Paired")

## -----------------------------------------------------------------------------
sessionInfo()

