## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----eval=FALSE---------------------------------------------------------------
# if (!require("BiocManager", quietly = TRUE))
#     install.packages("BiocManager")
# 
# BiocManager::install("delocal")

## ----eval=FALSE---------------------------------------------------------------
# if (!requireNamespace("devtools")) {
#   install.packages("devtools")
# }
# devtools::install_github("dasroy/delocal")

## ----example,message=FALSE,warning=FALSE--------------------------------------
library(DELocal)
count_matrix <- as.matrix(read.table(file = system.file("extdata", 
                                              "tooth_RNASeq_counts.txt", 
                                              package = "DELocal")))
colData <- data.frame(condition=gsub("\\..*",x=colnames(count_matrix),
                                     replacement = ""))

## ----message=FALSE,warning=FALSE----------------------------------------------
gene_location <- read.table(file = system.file("extdata", 
                                              "gene_location.txt", 
                                              package = "DELocal"))
head(gene_location)

## ----eval=TRUE----------------------------------------------------------------
library(biomaRt)
gene_attributes <- c("ensembl_gene_id", "start_position", "chromosome_name")
ensembl_ms_mart <- useMart(biomart="ENSEMBL_MART_ENSEMBL",
                           dataset="mmusculus_gene_ensembl", host="www.ensembl.org")
gene_location_sample <- getBM(attributes=gene_attributes, mart=ensembl_ms_mart,
                       verbose = FALSE)
rownames(gene_location_sample) <- gene_location_sample$ensembl_gene_id

## ----message=FALSE,warning=FALSE,error=FALSE----------------------------------
library(SummarizedExperiment)
smrExpt <- SummarizedExperiment(assays=list(counts=count_matrix),
                                                      rowData = gene_location, 
                                                      colData=colData)
smrExpt

## ----message=FALSE,warning=FALSE,error=FALSE----------------------------------
library(dplyr)
x_genes <- SummarizedExperiment::rowData(smrExpt) %>% 
    as.data.frame() %>% 
    filter(chromosome_name=="X") %>% rownames() 

DELocal_result <- DELocal(pSmrExpt = smrExpt[x_genes,],
                         nearest_neighbours = 5,pDesign = ~ condition,
                         pValue_cut = 0.05)

head(round(DELocal_result,digits = 9))

## ----message=FALSE,warning=FALSE,error=FALSE----------------------------------
DELocal::plotNeighbourhood(pSmrExpt = smrExpt, pGene_id = "ENSMUSG00000059401", 
                           pNearest_neighbours=5, pDesign = ~ condition)$plot

## ----message=FALSE,warning=FALSE,error=FALSE----------------------------------
gene_location_dynamicNeighbourhood <- read.csv(system.file("extdata", "Mouse_TAD_boundaries.csv",
                             package = "DELocal"))

rownames(gene_location_dynamicNeighbourhood) <- 
    gene_location_dynamicNeighbourhood$ensembl_gene_id

# rename the columns as required by DELocal
colnames(gene_location_dynamicNeighbourhood)[4:5] <- c("neighbors_start",
                                                       "neighbors_end")
common_genes <- intersect(rownames(count_matrix),
                          rownames(gene_location_dynamicNeighbourhood) )

smrExpt_dynamicNeighbour <-
    SummarizedExperiment::SummarizedExperiment(
        assays = list(counts = count_matrix[common_genes,]),
        rowData = gene_location_dynamicNeighbourhood[common_genes, ],
        colData = colData
    )

## Selecting only chromosome 1 genes to reduce the runtime                                                      
# one_genes <- SummarizedExperiment::rowData(smrExpt_dynamicNeighbour) %>% 
#     as.data.frame() %>% 
#     filter(chromosome_name=="1") %>% rownames() 

DELocal_result_tad <- DELocal(pSmrExpt = smrExpt_dynamicNeighbour[x_genes,], 
                         nearest_neighbours = 5,pDesign = ~ condition,
                         pValue_cut = 0.05, pLogFold_cut = 0)

head(DELocal_result_tad)

## ----echo=FALSE---------------------------------------------------------------
sessionInfo()

